﻿using System;
using System.Collections.Generic;
using System.Text;

namespace IndianHealthService.BMXNet.Model
{
    /// <summary>
    /// Context interface provides the current status and related events of the
    /// patient/visit context within an application, or the EHR.
    /// </summary>
    /// <remarks>
    /// Application developers use the WinFramework and ChangableContext to modify the context.  Within
    /// the EHR/VueCentric framework, the Context interface is a basic wrapper of some of the EHR facilties.
    /// </remarks>
    public interface Context
    {

        /// <summary>
        /// The most important aspect of the Context interface is the Changed event that allows
        /// the receiver of the event know when the user is changing patient and visit.  Applications
        /// need to change the data displayed to represent the currnet Patient and optional visit.  The
        /// Changed event occurs after there is confirmation that the change has not been vetoed or the
        /// veto has been overridden.  See Changing event.
        /// </summary>
        /// <example>
        /// Users of the Context object should hook the event ASAP.  The 
        /// <code>
        ///public void Startup()
        /// {
        ///     if (this.Context != null)
        ///     {
        ///         this.Context.Changed += new EventHandler&lt;ContextChangedArgs&gt;(Context_Changed);
        /// }
        ///</code>
        /// The context can be obtained if your control or component is a LocalSessionConsumer and if so then
        /// define the Context property as follows:
        /// <code>
        /// public Context Context
        /// {
        ///     get { return this.LocalSession == null ? null : this.LocalSession.Context; }
        /// }
        /// </code>
        /// </example>
        event EventHandler<ContextChangedArgs> Changed;
        /// <summary>
        /// When the current patient or visit is being changed by the user, every subscriber to the
        /// Changing method has the option to "Veto" the event by Cancelling it:
        /// <code>
        ///public void Startup()
        /// {
        ///     if (this.Context != null)
        ///     {
        ///         this.Context.Changed += new EventHandler&lt;ContextChangedArgs&gt;(Context_Changed);
        ///         this.Context.Changing += new EventHandler&lt;ContextChangedArgs&gt;(Context_Changing);
        /// }
        /// 
        /// void Context_Changing(object sender, ContextChangingArgs e)
        /// {
        ///     e.Cancel = needToFinishSomethingElseBeforeChangingContext;                
        /// }
        ///</code>
        /// However, a Veto is like a U.S. presential Veto and that it can be overridden.  The WinFramework
        /// or EHR can override any Veto so all components (even if they veto changes) must be prepared to
        /// have the context changed at anytime.
        /// </summary>
        event EventHandler<ContextChangingArgs> Changing;
            
        /// <summary>
        /// The currently selected patient, or null.  The value of the current patient is also in <seealso cref="ContextChangedArgs"/>.
        /// The concept is that the context is set by external forces, ususally the user.  Therefore there is no
        /// way to set the patient here.
        /// </summary>
        /// <remarks>
        /// Remember to refresh your user interface based on the new selection to the information your application displays
        /// or edits is that same as the current context.
        /// </remarks>
        Patient Patient { get; }
        /// <summary>
        /// The currently selected visit, or null.   The value of the current visit is also in ContextChangedArgs.
        /// The concept is that the context is set by external forces, ususally the user.  Therefore there is no
        /// way to set the visit here.
        /// </summary>
        /// <remarks>
        /// In the EHR, a visit stub can be created.  A visit stub does not create a visit in RPMS until some data
        /// is entered.  IsVisit will return true for a visit stub but is it your responbilty to use HasUnlockedVisit or IsStub 
        /// on the Visit to make sure you have a visit to edit with.
        /// Remember to refresh your user interface based on the new selection to the information your application displays
        /// or edits is that same as the current context.   
        /// </remarks>
        Visit Visit { get; }

        /// <summary>
        /// Answer true/false if there is a patient currently selected.  This is a convenience method.
        /// </summary>
        bool HasPatient { get; }
        /// <summary>
        /// Answer true/false if there is a patient currently selected.  This is a convenience method.
        /// </summary>
        bool HasVisit { get; }
        /// <summary>
        /// Answer true if there is a visit and if it is unlocked.  The EHR/VueCentric has a notion of the
        /// locked visit, a visit that can no longer be visited.  The EHR/VueCetnric has has a visit stub, a visit that
        /// has been create on the desktop but not in RPMS.  If a visit is a visit stub, it is not considered unlocked.
        /// </summary>
        bool HasUnlockedVisit { get; }        

        /// <summary>
        /// The user that established the connection to BMX either through the WinFramework dialogs or access/verify codes, or
        /// a user established through attach tho the EHR/VueCentric with the EhrFramework.
        /// </summary>
        User User { get; }
    }
}
